/* $Header: /cvsroot/osrs/libtiff/libtiff/tif_print.c,v 1.5 2001/03/02 04:59:52 warmerda Exp $ */

/*
 * Copyright (c) 1988-1997 Sam Leffler
 * Copyright (c) 1991-1997 Silicon Graphics, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Sam Leffler and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Sam Leffler and Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 * IN NO EVENT SHALL SAM LEFFLER OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF 
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 */


/*
 * TIFF Library.
 *
 * Directory Printing Support
 */

#ifndef _MT
#define _MT
#endif

#include "tiffiop.h"
#include <stdio.h>



#include "ctype.h"

static const char *photoNames[] = {
    "min-is-white",				/* PHOTOMETRIC_MINISWHITE */
    "min-is-black",				/* PHOTOMETRIC_MINISBLACK */
    "RGB color",				/* PHOTOMETRIC_RGB */
    "palette color (RGB from colormap)",	/* PHOTOMETRIC_PALETTE */
    "transparency mask",			/* PHOTOMETRIC_MASK */
    "separated",				/* PHOTOMETRIC_SEPARATED */
    "YCbCr",					/* PHOTOMETRIC_YCBCR */
    "7 (0x7)",
    "CIE L*a*b*",				/* PHOTOMETRIC_CIELAB */
};
#define	NPHOTONAMES	(sizeof (photoNames) / sizeof (photoNames[0]))

static const char *orientNames[] = {
    "0 (0x0)",
    "row 0 top, col 0 lhs",			/* ORIENTATION_TOPLEFT */
    "row 0 top, col 0 rhs",			/* ORIENTATION_TOPRIGHT */
    "row 0 bottom, col 0 rhs",			/* ORIENTATION_BOTRIGHT */
    "row 0 bottom, col 0 lhs",			/* ORIENTATION_BOTLEFT */
    "row 0 lhs, col 0 top",			/* ORIENTATION_LEFTTOP */
    "row 0 rhs, col 0 top",			/* ORIENTATION_RIGHTTOP */
    "row 0 rhs, col 0 bottom",			/* ORIENTATION_RIGHTBOT */
    "row 0 lhs, col 0 bottom",			/* ORIENTATION_LEFTBOT */
};
#define	NORIENTNAMES	(sizeof (orientNames) / sizeof (orientNames[0]))




static char * _TIFFsprintAscii(char * p,const char* cp)
{
		const char* tp;

	for (; *cp != '\0'; cp++) {


		if(isprint(*cp)) {
   *p++=*cp;
			continue;
		}
		for (tp = "\tt\bb\rr\nn\vv"; *tp; tp++)
			if (*tp++ == *cp)
				break;
		if (*tp)
			p+=sprintf(p, "\\%c", *tp);
		else
			p+=sprintf(p, "\\%03o", *cp & 0xff);
	}

 return(p);
}





static char * _TIFFsprintAsciiTag(char * p, const char* name, const char* value)
{
	p+=sprintf(p, "  %s: \"", name);
	p=_TIFFsprintAscii(p, value);
	p+=sprintf(p, "\"\n");

 return(p);
}





/*
 * Print the contents of the current directory
 * to the specified stdio file stream.
 */
void
TIFFSPrintDirectory(TIFF* tif, char * p, long flags)
{
	register TIFFDirectory *td;
	char *sep;
	uint16 i;
	long l, n;

	p+=sprintf(p, "TIFF Directory at offset 0x%lx\n", tif->tif_diroff);
	td = &tif->tif_dir;
	if (TIFFFieldSet(tif,FIELD_SUBFILETYPE)) {
		p+=sprintf(p, "Sub Type:");
		sep = " ";
		if (td->td_subfiletype & FILETYPE_REDUCEDIMAGE) {
			p+=sprintf(p,"%sreduced-resolution image", sep);
			sep = "/";
		}
		if (td->td_subfiletype & FILETYPE_PAGE) {
			p+=sprintf(p,"%smulti-page document", sep);
			sep = "/";
		}
		if (td->td_subfiletype & FILETYPE_MASK)
			p+=sprintf(p, "%stransparency mask", sep);
		 p+=sprintf(p, " (%lu = 0x%lx)\n",
		    (long) td->td_subfiletype, (long) td->td_subfiletype);
	}
	if (TIFFFieldSet(tif,FIELD_IMAGEDIMENSIONS)) {
		p+=sprintf(p, "  Image Width: %lu Image Length: %lu",
		    (u_long) td->td_imagewidth, (u_long) td->td_imagelength);
		if (TIFFFieldSet(tif,FIELD_IMAGEDEPTH))
		p+=sprintf(p, " Image Depth: %lu",
			    (u_long) td->td_imagedepth);
		p+=sprintf(p,"\n");
	}

 	/* Begin Pixar */
#ifdef NEVER
 	if (TIFFFieldSet(tif,FIELD_IMAGEFULLWIDTH) ||
 	    TIFFFieldSet(tif,FIELD_IMAGEFULLLENGTH)) {
	  p+=sprintf(p,"  Pixar Full Image Width: %lu Full Image Length: %lu\n",
		  (u_long) td->td_imagefullwidth,
		  (u_long) td->td_imagefulllength);
 	}
 	if (TIFFFieldSet(tif,FIELD_TEXTUREFORMAT))
	  p=_TIFFsprintAsciiTag(p, "Texture Format", td->td_textureformat);
 	if (TIFFFieldSet(tif,FIELD_WRAPMODES))
	  p=_TIFFsprintAsciiTag(p, "Texture Wrap Modes", td->td_wrapmodes);
 	if (TIFFFieldSet(tif,FIELD_FOVCOT))
	  p+=sprintf(p, "  Field of View Cotangent: %g\n", td->td_fovcot);
	if (TIFFFieldSet(tif,FIELD_MATRIX_WORLDTOSCREEN)) {
	  typedef float	Matrix[4][4];
	  Matrix*		m = (Matrix*)td->td_matrixWorldToScreen;
	  
	  p+=sprintf(p, "  Matrix NP:\n\t%g %g %g %g\n\t%g %g %g %g\n\t%g %g %g %g\n\t%g %g %g %g\n",
		  (*m)[0][0], (*m)[0][1], (*m)[0][2], (*m)[0][3],
		  (*m)[1][0], (*m)[1][1], (*m)[1][2], (*m)[1][3],
		  (*m)[2][0], (*m)[2][1], (*m)[2][2], (*m)[2][3],
		  (*m)[3][0], (*m)[3][1], (*m)[3][2], (*m)[3][3]);
 	}
 	if (TIFFFieldSet(tif,FIELD_MATRIX_WORLDTOCAMERA)) {
	  typedef float	Matrix[4][4];
	  Matrix*		m = (Matrix*)td->td_matrixWorldToCamera;
	  
	  p+=sprintf(p, "  Matrix Nl:\n\t%g %g %g %g\n\t%g %g %g %g\n\t%g %g %g %g\n\t%g %g %g %g\n",
		  (*m)[0][0], (*m)[0][1], (*m)[0][2], (*m)[0][3],
		  (*m)[1][0], (*m)[1][1], (*m)[1][2], (*m)[1][3],
		  (*m)[2][0], (*m)[2][1], (*m)[2][2], (*m)[2][3],
		  (*m)[3][0], (*m)[3][1], (*m)[3][2], (*m)[3][3]);
 	}
 	/* End Pixar */
#endif	
	if (TIFFFieldSet(tif,FIELD_TILEDIMENSIONS)) {
		p+=sprintf(p, "  Tile Width: %lu Tile Length: %lu",
		    (u_long) td->td_tilewidth, (u_long) td->td_tilelength);
		if (TIFFFieldSet(tif,FIELD_TILEDEPTH))
		p+=sprintf(p, " Tile Depth: %lu",
			    (u_long) td->td_tiledepth);
		p+=sprintf(p, "\n");
	}
	if (TIFFFieldSet(tif,FIELD_RESOLUTION)) {
		p+=sprintf(p, "  Resolution: %g, %g",
		    td->td_xresolution, td->td_yresolution);
		if (TIFFFieldSet(tif,FIELD_RESOLUTIONUNIT)) {
			switch (td->td_resolutionunit) {
			case RESUNIT_NONE:
				p+=sprintf(p, " (unitless)");
				break;
			case RESUNIT_INCH:
				p+=sprintf(p, " pixels/inch");
				break;
			case RESUNIT_CENTIMETER:
				p+=sprintf(p, " pixels/cm");
				break;
			default:
				p+=sprintf(p, " (unit %u = 0x%x)",
				    td->td_resolutionunit,
				    td->td_resolutionunit);
				break;
			}
		}
		p+=sprintf(p, "\n");
	}
	if (TIFFFieldSet(tif,FIELD_POSITION))
		p+=sprintf(p, "  Position: %g, %g\n",
		    td->td_xposition, td->td_yposition);
	if (TIFFFieldSet(tif,FIELD_BITSPERSAMPLE))
		p+=sprintf(p, "  Bits/Sample: %u\n", td->td_bitspersample);
	if (TIFFFieldSet(tif,FIELD_SAMPLEFORMAT)) {
		p+=sprintf(p, "  Sample Format: ");
		switch (td->td_sampleformat) {
		case SAMPLEFORMAT_VOID:
			p+=sprintf(p, "void\n");
			break;
		case SAMPLEFORMAT_INT:
			p+=sprintf(p, "signed integer\n");
			break;
		case SAMPLEFORMAT_UINT:
			p+=sprintf(p, "unsigned integer\n");
			break;
		case SAMPLEFORMAT_IEEEFP:
			p+=sprintf(p, "IEEE floating point\n");
			break;
		case SAMPLEFORMAT_COMPLEXINT:
			p+=sprintf(p, "complex signed integer\n");
			break;
		case SAMPLEFORMAT_COMPLEXIEEEFP:
			p+=sprintf(p, "complex IEEE floating point\n");
			break;
		default:
			p+=sprintf(p, "%u (0x%x)\n",
			    td->td_sampleformat, td->td_sampleformat);
			break;
		}
	}
	if (TIFFFieldSet(tif,FIELD_COMPRESSION)) {
		const TIFFCodec* c = TIFFFindCODEC(td->td_compression);
		p+=sprintf(p, "  Compression: ");
		if (c)
			p+=sprintf(p, "%s\n", c->name);
		else
			p+=sprintf(p, "%u (0x%x)\n",
			    td->td_compression, td->td_compression);
	}
	if (TIFFFieldSet(tif,FIELD_PHOTOMETRIC)) {
		p+=sprintf(p, "  Photo Interp: ");
		if (td->td_photometric < NPHOTONAMES)
			p+=sprintf(p, "%s\n", photoNames[td->td_photometric]);
		else {
			switch (td->td_photometric) {
			case PHOTOMETRIC_LOGL:
				p+=sprintf(p, "CIE Log2(L)\n");
				break;
			case PHOTOMETRIC_LOGLUV:
				p+=sprintf(p, "CIE Log2(L) (u',v')\n");
				break;
			default:
				p+=sprintf(p, "%u (0x%x)\n",
				    td->td_photometric, td->td_photometric);
				break;
			}
		}
	}
	if (TIFFFieldSet(tif,FIELD_EXTRASAMPLES) && td->td_extrasamples) {
		p+=sprintf(p, "  Extra Samples: %u<", td->td_extrasamples);
		sep = "";
		for (i = 0; i < td->td_extrasamples; i++) {
			switch (td->td_sampleinfo[i]) {
			case EXTRASAMPLE_UNSPECIFIED:
				p+=sprintf(p, "%sunspecified", sep);
				break;
			case EXTRASAMPLE_ASSOCALPHA:
				p+=sprintf(p, "%sassoc-alpha", sep);
				break;
			case EXTRASAMPLE_UNASSALPHA:
				p+=sprintf(p, "%sunassoc-alpha", sep);
				break;
			default:
				p+=sprintf(p, "%s%u (0x%x)", sep,
				    td->td_sampleinfo[i], td->td_sampleinfo[i]);
				break;
			}
			sep = ", ";
		}
		p+=sprintf(p, ">\n");
	}
	if (TIFFFieldSet(tif,FIELD_STONITS)) {
		p+=sprintf(p, "  Sample to Nits conversion factor: %.4e\n",
				td->td_stonits);
	}
#ifdef CMYK_SUPPORT
	if (TIFFFieldSet(tif,FIELD_INKSET)) {
		p+=sprintf(p, "  Ink Set: ");
		switch (td->td_inkset) {
		case INKSET_CMYK:
			p+=sprintf(p, "CMYK\n");
			break;
		default:
			p+=sprintf(p, "%u (0x%x)\n",
			    td->td_inkset, td->td_inkset);
			break;
		}
	}
	if (TIFFFieldSet(tif,FIELD_INKNAMES)) {
		char* cp;
		p+=sprintf(p, "  Ink Names: ");
		i = td->td_samplesperpixel;
		sep = "";
		for (cp = td->td_inknames; i > 0; cp = strchr(cp,'\0')+1, i--) {
			p+=sprintf(p, "%s", sep);
			p=_TIFFsprintAscii(p, cp);
			sep = ", ";
		}
	}
	if (TIFFFieldSet(tif,FIELD_NUMBEROFINKS))
		p+=sprintf(p, " Number of Inks: %u\n", td->td_ninks);
	if (TIFFFieldSet(tif,FIELD_DOTRANGE))
		p+=sprintf(p, "  Dot Range: %u-%u\n",
		    td->td_dotrange[0], td->td_dotrange[1]);
//	if (TIFFFieldSet(tif,FIELD_TARGETPRINTER))
//		p=_TIFFsprintAsciiTag(p, "Target Printer", td->td_targetprinter);
#endif
	if (TIFFFieldSet(tif,FIELD_THRESHHOLDING)) {
		p+=sprintf(p, "  Thresholding: ");
		switch (td->td_threshholding) {
		case THRESHHOLD_BILEVEL:
			p+=sprintf(p, "bilevel art scan\n");
			break;
		case THRESHHOLD_HALFTONE:
			p+=sprintf(p, "halftone or dithered scan\n");
			break;
		case THRESHHOLD_ERRORDIFFUSE:
			p+=sprintf(p, "error diffused\n");
			break;
		default:
			p+=sprintf(p, "%u (0x%x)\n",
			    td->td_threshholding, td->td_threshholding);
			break;
		}
	}
	if (TIFFFieldSet(tif,FIELD_FILLORDER)) {
		p+=sprintf(p, "  FillOrder: ");
		switch (td->td_fillorder) {
		case FILLORDER_MSB2LSB:
			p+=sprintf(p, "msb-to-lsb\n");
			break;
		case FILLORDER_LSB2MSB:
			p+=sprintf(p, "lsb-to-msb\n");
			break;
		default:
			p+=sprintf(p, "%u (0x%x)\n",
			    td->td_fillorder, td->td_fillorder);
			break;
		}
	}
#ifdef YCBCR_SUPPORT
	if (TIFFFieldSet(tif,FIELD_YCBCRSUBSAMPLING))
		p+=sprintf(p, "  YCbCr Subsampling: %u, %u\n",
		    td->td_ycbcrsubsampling[0], td->td_ycbcrsubsampling[1]);
	if (TIFFFieldSet(tif,FIELD_YCBCRPOSITIONING)) {
		p+=sprintf(p, "  YCbCr Positioning: ");
		switch (td->td_ycbcrpositioning) {
		case YCBCRPOSITION_CENTERED:
			p+=sprintf(p, "centered\n");
			break;
		case YCBCRPOSITION_COSITED:
			p+=sprintf(p, "cosited\n");
			break;
		default:
			p+=sprintf(p, "%u (0x%x)\n",
			    td->td_ycbcrpositioning, td->td_ycbcrpositioning);
			break;
		}
	}
	if (TIFFFieldSet(tif,FIELD_YCBCRCOEFFICIENTS))
		p+=sprintf(p, "  YCbCr Coefficients: %g, %g, %g\n",
		    td->td_ycbcrcoeffs[0],
		    td->td_ycbcrcoeffs[1],
		    td->td_ycbcrcoeffs[2]);
#endif
	if (TIFFFieldSet(tif,FIELD_HALFTONEHINTS))
		p+=sprintf(p, "  Halftone Hints: light %u dark %u\n",
		    td->td_halftonehints[0], td->td_halftonehints[1]);
//	if (TIFFFieldSet(tif,FIELD_ARTIST))
//		p=_TIFFsprintAsciiTag(p, "Artist", td->td_artist);
//	if (TIFFFieldSet(tif,FIELD_DATETIME))
//		p=_TIFFsprintAsciiTag(p, "Date & Time", td->td_datetime);
//	if (TIFFFieldSet(tif,FIELD_HOSTCOMPUTER))
//		p=_TIFFsprintAsciiTag(p, "Host Computer", td->td_hostcomputer);
//	if (TIFFFieldSet(tif,FIELD_SOFTWARE))
//		p=_TIFFsprintAsciiTag(p, "Software", td->td_software);
//	if (TIFFFieldSet(tif,FIELD_COPYRIGHT))
//		p=_TIFFsprintAsciiTag(p, "Copyright", td->td_copyright);
//	if (TIFFFieldSet(tif,FIELD_DOCUMENTNAME))
//		p=_TIFFsprintAsciiTag(p, "Document Name", td->td_documentname);
//	if (TIFFFieldSet(tif,FIELD_IMAGEDESCRIPTION))
//		p=_TIFFsprintAsciiTag(p, "Image Description", td->td_imagedescription);
//	if (TIFFFieldSet(tif,FIELD_MAKE))
//		p=_TIFFsprintAsciiTag(p, "Make", td->td_make);
//	if (TIFFFieldSet(tif,FIELD_MODEL))
//		p=_TIFFsprintAsciiTag(p, "Model", td->td_model);
	if (TIFFFieldSet(tif,FIELD_ORIENTATION)) {
		p+=sprintf(p, "  Orientation: ");
		if (td->td_orientation < NORIENTNAMES)
			p+=sprintf(p, "%s\n", orientNames[td->td_orientation]);
		else
			p+=sprintf(p, "%u (0x%x)\n",
			    td->td_orientation, td->td_orientation);
	}
	if (TIFFFieldSet(tif,FIELD_SAMPLESPERPIXEL))
		p+=sprintf(p, "  Samples/Pixel: %u\n", td->td_samplesperpixel);
	if (TIFFFieldSet(tif,FIELD_ROWSPERSTRIP)) {
		p+=sprintf(p, "  Rows/Strip: ");
		if (td->td_rowsperstrip == (uint32) -1)
			p+=sprintf(p, "(infinite)\n");
		else
			p+=sprintf(p, "%lu\n", (u_long) td->td_rowsperstrip);
	}
	if (TIFFFieldSet(tif,FIELD_MINSAMPLEVALUE))
		p+=sprintf(p, "  Min Sample Value: %u\n", td->td_minsamplevalue);
	if (TIFFFieldSet(tif,FIELD_MAXSAMPLEVALUE))
		p+=sprintf(p, "  Max Sample Value: %u\n", td->td_maxsamplevalue);
	if (TIFFFieldSet(tif,FIELD_SMINSAMPLEVALUE))
		p+=sprintf(p, "  SMin Sample Value: %g\n",
		    td->td_sminsamplevalue);
	if (TIFFFieldSet(tif,FIELD_SMAXSAMPLEVALUE))
		p+=sprintf(p, "  SMax Sample Value: %g\n",
		    td->td_smaxsamplevalue);
	if (TIFFFieldSet(tif,FIELD_PLANARCONFIG)) {
		p+=sprintf(p, "  Planar Configuration: ");
		switch (td->td_planarconfig) {
		case PLANARCONFIG_CONTIG:
			p+=sprintf(p, "single image plane\n");
			break;
		case PLANARCONFIG_SEPARATE:
			p+=sprintf(p, "separate image planes\n");
			break;
		default:
			p+=sprintf(p, "%u (0x%x)\n",
			    td->td_planarconfig, td->td_planarconfig);
			break;
		}
	}
//	if (TIFFFieldSet(tif,FIELD_PAGENAME))
//		p=_TIFFsprintAsciiTag(p, "Page Name", td->td_pagename);
	if (TIFFFieldSet(tif,FIELD_PAGENUMBER))
		p+=sprintf(p, "  Page Number: %u-%u\n",
		    td->td_pagenumber[0], td->td_pagenumber[1]);
	if (TIFFFieldSet(tif,FIELD_COLORMAP)) {
		p+=sprintf(p, "  Color Map: ");
		if (flags & TIFFPRINT_COLORMAP) {
			p+=sprintf(p, "\n");
			n = 1L<<td->td_bitspersample;
			for (l = 0; l < n; l++)
				p+=sprintf(p, "   %5lu: %5u %5u %5u\n",
				    l,
				    td->td_colormap[0][l],
				    td->td_colormap[1][l],
				    td->td_colormap[2][l]);
		} else
			p+=sprintf(p, "(present)\n");
	}
#ifdef COLORIMETRY_SUPPORT
	if (TIFFFieldSet(tif,FIELD_WHITEPOINT))
		p+=sprintf(p, "  White Point: %g-%g\n",
		    td->td_whitepoint[0], td->td_whitepoint[1]);
//	if (TIFFFieldSet(tif,FIELD_PRIMARYCHROMAS))
//		p+=sprintf(p, "  Primary Chromaticities: %g,%g %g,%g %g,%g\n",
//		    td->td_primarychromas[0], td->td_primarychromas[1],
//		    td->td_primarychromas[2], td->td_primarychromas[3],
//		    td->td_primarychromas[4], td->td_primarychromas[5]);
	if (TIFFFieldSet(tif,FIELD_REFBLACKWHITE)) {
		p+=sprintf(p, "  Reference Black/White:\n");
		for (i = 0; i < td->td_samplesperpixel; i++)
			p+=sprintf(p, "    %2d: %5g %5g\n",
			    i,
			    td->td_refblackwhite[2*i+0],
			    td->td_refblackwhite[2*i+1]);
	}
	if (TIFFFieldSet(tif,FIELD_TRANSFERFUNCTION)) {
		p+=sprintf(p, "  Transfer Function: ");
		if (flags & TIFFPRINT_CURVES) {
			p+=sprintf(p, "\n");
			n = 1L<<td->td_bitspersample;
			for (l = 0; l < n; l++) {
				p+=sprintf(p, "    %2lu: %5u",
				    l, td->td_transferfunction[0][l]);
				for (i = 1; i < td->td_samplesperpixel; i++)
					p+=sprintf(p, " %5u",
					    td->td_transferfunction[i][l]);
				*p++='\n';
			}
		} else
			p+=sprintf(p, "(present)\n");
	}
#endif
#ifdef ICC_SUPPORT
	if (TIFFFieldSet(tif,FIELD_ICCPROFILE))
		p+=sprintf(p, "  ICC Profile: <present>, %lu bytes\n",
		    (u_long) td->td_profileLength);
#endif
#ifdef PHOTOSHOP_SUPPORT
 	if (TIFFFieldSet(tif,FIELD_PHOTOSHOP))
 		p+=sprintf(p, "  Photoshop Data: <present>, %lu bytes\n",
 		    (u_long) td->td_photoshopLength);
#endif
#ifdef IPTC_SUPPORT
 	if (TIFFFieldSet(tif,FIELD_RICHTIFFIPTC))
 		p+=sprintf(p, "  RichTIFFIPTC Data: <present>, %lu bytes\n",
 		    (u_long) td->td_richtiffiptcLength);
#endif
#if SUBIFD_SUPPORT
	if (TIFFFieldSet(tif, FIELD_SUBIFD)) {
		p+=sprintf(p, "  SubIFD Offsets:");
		for (i = 0; i < td->td_nsubifd; i++)
			p+=sprintf(p, " %5lu", (long) td->td_subifd[i]);
		*p++='\n';
	}
#endif

//	if(tif->tif_printdir)
//		(*tif->tif_printdir)(tif, fd, flags);

	if ((flags & TIFFPRINT_STRIPS) &&
	    TIFFFieldSet(tif,FIELD_STRIPOFFSETS)) {
		tstrip_t s;

		p+=sprintf(p, "  %lu %s:\n",
		    (long) td->td_nstrips,
		    isTiled(tif) ? "Tiles" : "Strips");
		for (s = 0; s < td->td_nstrips; s++)
			p+=sprintf(p, "    %3lu: [%8lu, %8lu]\n",
			    (u_long) s,
			    (u_long) td->td_stripoffset[s],
			    (u_long) td->td_stripbytecount[s]);
	}
}

