/*->c.quant */


#include "stdafx.h"

#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <locale.h>
#include <math.h> 

#include "os.h"
#include "wimp.h"
#include "wimpt.h"
#include "werr.h"
#include "transform.h"
#include "flex.h"
#include "bbc.h"

#include "wos.h"
#include "err.h"
#include "task.h"
#include "xext.h"
#include "poll.h"
#include "temp.h"
#include "scrap.h"
#include "fsx.h"
#include "bf.h"
#include "etc.h"
#include "dbhi.h"
#include "key.h"
#include "mlo.h"
#include "units.h"
#include "bits.h"
#include "conf.h"
#include "colour.h"
#include "redraw.h"

#include "str.h"
#include "constants.h"

#include "reslink.h"
#include "file.h"
#include "view.h"
#include "viewr.h"
#include "im.h"


#include "quant.h"




#define HUGE HUGE_VAL


/* 
 * The colorquant() routine has been tested on an Iris 4D70 workstation,
 * a Sun 3/60 workstation, and (to some extent), a Macintosh.
 * 
 * Calls to bzero() may have to be replaced with the appropriate thing on
 * your system.  (bzero(ptr, len) writes 'len' 0-bytes starting at the location
 * pointed to by ptr.)
 * 
 * Although I've tried to avoid integer overflow problems where ever possible,
 * it's likely I've missed a spot where an 'int' should really be a 'long'.
 * (On the machine this was developed on, an int == long == 32 bits.)
 * 
 * Note that it's quite easy to optimize this code for a given value for
 * 'bits'.  In addition, if you assume bits is small and
 * that the total number of pixels is relatively small, there are several
 * places that integer arithmetic may be substituted for floating-point.
 * One such place is the loop in boxstat -- mean and var need not necessary
 * be floats.
 * 
 * As things stand, the maximum number of colors to which an image may
 * be quantized is 256.  This limit may be overcome by changing rgbmap and
 * colormap from pointers to characters to pointers to something larger.
 */

/*
 * Maximum number of colormap entries.  To make larger than 2^8, the rgbmap
 * type will have to be changed from unsigned chars to something larger.
 */
#define MAXCOLORS               256
/*
 * Value corrersponding to full intensity in colormap.  The values placed
 * in the colormap are scaled to be between zero and this number.  Note
 * that anything larger than 255 is going to lead to problems, as the
 * colormap is declared as an unsigned char.
 */
#define FULLINTENSITY           255


/*
 * Readability constants.
 */

#define REDI            0       
#define GREENI          1
#define BLUEI           2       

typedef struct 
{
        int             low[3];
        int             high[3];                /* Box extent */
        double          weightedvar;            /* weighted variance */
        double          mean[3];                /* centroid */
        unsigned int    weight;                 /* # of pixels in box */
        unsigned int    freq[3][MAXCOLORS];     /* Projected frequencies */
} Box;



static unsigned int   * Histogram;              /* image histogram */
static unsigned int     NPixels;                /* # of pixels in an image*/
static unsigned int     SumPixels;              /* total # of pixels */
static unsigned int     Bits;                   /* # significant input bits */
static unsigned int     ColormaxI;              /* # of colors, 2^Bits */
static Box            * Boxes;                  /* Array of color boxes. */



/*
 * Compute mean and weighted variance of the given box.
 */

static void boxstat(Box * box)
{
 int i, color;
 unsigned int *freq;
 double mean, var;
 double temp;

 if(box->weight==0) 
 {
  box->weightedvar=0;
  return;
 }

 box->weightedvar=0.;
 for(color=0;color<3;color++) 
 {
  var=mean=0;
  i=box->low[color];
  freq=&box->freq[color][i];
  for(;i<box->high[color];i++,freq++) 
  {
   mean += i * *freq;
   var += i*i* *freq;
  }
  box->mean[color] = mean / (double)box->weight;

  temp=var-box->mean[color]*box->mean[color]*(double)box->weight;

  if(color==REDI) temp*=3;
  else
  if(color==GREENI) temp*=10;

  box->weightedvar+=temp;
 }

 box->weightedvar/=(SumPixels*14);
}




/*
 * Update projected frequency arrays for two boxes which used to be
 * a single box.
 */

static void updatefreq(Box * box1,Box * box2)
{
 unsigned int myfreq, *h;
 int b, g, r;
 int roff;

 memset(box1->freq[0],0,ColormaxI*sizeof(unsigned int));
 memset(box1->freq[1],0,ColormaxI*sizeof(unsigned int));
 memset(box1->freq[2],0,ColormaxI*sizeof(unsigned int)); 

 for(r=box1->low[0];r<box1->high[0];r++) 
 {
  roff=r<<Bits;
  for(g=box1->low[1];g<box1->high[1];g++) 
  {
   b=box1->low[2];
   h=Histogram+(((roff|g)<<Bits)|b);
   for(;b<box1->high[2];b++) 
   {
    if((myfreq=*h++)==0) continue;

    box1->freq[0][r] += myfreq;
    box1->freq[1][g] += myfreq;
    box1->freq[2][b] += myfreq;
    box2->freq[0][r] -= myfreq;
    box2->freq[1][g] -= myfreq;
    box2->freq[2][b] -= myfreq;
   }
  }
 }
}






/*
 * Compute the 'optimal' cutpoint for the given box along the axis
 * indcated by 'color'.  Store the boxes which result from the cut
 * in newbox1 and newbox2.
 */
static int findcut(Box * box,int color,Box * newbox1,Box * newbox2)
{
 double u, v, max;
 int i, maxindex, minindex, cutpoint;
 unsigned int optweight, curweight;

 if(box->low[color]+1==box->high[color])
 {
  return(FALSE);   /* Cannot be cut. */
 }

 minindex=(int)((box->low[color] + box->mean[color]) * 0.5);
 maxindex=(int)((box->mean[color] + box->high[color]) * 0.5);

 cutpoint = minindex;
 optweight = box->weight;

 curweight = 0;


 for(i=box->low[color];i<minindex;i++) curweight+=box->freq[color][i];
 u=0.;
 max=-1;


 for(i=minindex;i<=maxindex;i++) 
 {
  curweight+=box->freq[color][i];
  if(curweight==box->weight) break;
  u+=(double)(i*box->freq[color][i])/(double)box->weight;
  v=((double)curweight/(double)(box->weight-curweight))*(box->mean[color]-u)*(box->mean[color]-u);
  if(v>max) 
  {
   max=v;
   cutpoint=i;
   optweight=curweight;
  }
 }

 cutpoint++;
 *newbox1 = *newbox2 = *box;
 newbox1->weight = optweight;
 newbox2->weight -= optweight;
 newbox1->high[color] = cutpoint;
 newbox2->low[color] = cutpoint;
 updatefreq(newbox1, newbox2);
 boxstat(newbox1);
 boxstat(newbox2);

 return(TRUE);    /* Found cutpoint. */
}






/*
 * Cut the given box.  Returns TRUE if the box could be cut, FALSE otherwise.
 */

static int splitbox(Box * box,Box * newbox)
{
 int i;
 double totalvar[3];
 Box newboxes[3][2];

 if(box->weightedvar==0. || box->weight==0)
 {               /*
                  * Can't cut this box.
                  */
  return(FALSE);
 }

        /*
         * Find 'optimal' cutpoint along each of the red, green and blue
         * axes.  Sum the variances of the two boxes which would result
         * by making each cut and store the resultant boxes for 
         * (possible) later use.
         */

 for(i=0;i<3;i++) 
 {
  if(findcut(box,i,&newboxes[i][0],&newboxes[i][1]))
            totalvar[i]=newboxes[i][0].weightedvar+newboxes[i][1].weightedvar;
  else      totalvar[i]=HUGE;
 }

 /*
  * Find which of the three cuts minimized the total variance
  * and make that the 'real' cut.
  */



 if(totalvar[REDI]<=totalvar[GREENI] &&
            totalvar[REDI] <= totalvar[BLUEI]) 
 {
  *box=newboxes[REDI][0];
  *newbox=newboxes[REDI][1];
 } 
 else 
 if(totalvar[GREENI]<=totalvar[REDI] && totalvar[GREENI]<=totalvar[BLUEI]) 
 {
  *box=newboxes[GREENI][0];
  *newbox=newboxes[GREENI][1];
 } 
 else 
 {
  *box=newboxes[BLUEI][0];
  *newbox=newboxes[BLUEI][1];
 }

 return(TRUE);
}





/*
 * Return the number of the box in 'boxes' with the greatest variance.
 * Restrict the search to those boxes with indices between 0 and n-1.
 */

static int maxvar(Box * boxes,int n)
{
 int i, whichbox = 0;
 double max;

 max=-1;
 for(i=0;i<n;i++) 
 {
  if(boxes[i].weightedvar>max) 
  {
   max=boxes[i].weightedvar;
   whichbox=i;
  }
 }
 return(whichbox);
}






/*
 * Interatively cut the boxes.
 */

static int splitboxes(Box * boxes,int colors) 
{
 int curbox;

 boxes[0].low[REDI]=boxes[0].low[GREENI]=boxes[0].low[BLUEI]=0;
 boxes[0].high[REDI]=boxes[0].high[GREENI]=boxes[0].high[BLUEI]=ColormaxI;
 boxes[0].weight=SumPixels;

 boxstat(&boxes[0]);

 for(curbox=1;curbox<colors;curbox++) 
 {
  if(splitbox(&boxes[maxvar(boxes,curbox)],
                                           &boxes[curbox])==FALSE)break;
 }


 return(curbox);
}











/*
 * Compute the histogram of the image as well as the projected frequency
 * arrays for the first world-encompassing box.
 */

static os_error * calchist(imagestr * im,Box * box,int bits)
{
 os_error * err;
 int        x;
 int        y;
 int        xpix;
 int        ypix;
 int        bpp;
 int      * data;
 char     * p;
 int        mask;
 int        shift;
 int        c;
 int        index;
 char       rtab[256];
 char       gtab[256];
 char       btab[256];
 unsigned int *rf, *gf, *bf;
 int        i;
 int        rr;
 int        gg;
 int        bb;

 err=NULL;

 rf=box->freq[0];
 gf=box->freq[1];
 bf=box->freq[2];

 xpix=im->xpix;
 ypix=im->ypix;
 bpp=im->bpp;

 if(bpp<=8)
 {
  for(i=0;i<im->ipal.ncolours;i++)
  {
   c=im->ipal.word[i];
   rtab[i]=(char)(((c>> 8)&0xFF)>>(8-bits));
   gtab[i]=(char)(((c>>16)&0xFF)>>(8-bits));
   btab[i]=(char)(((c>>24)&0xFF)>>(8-bits));
  }
 }

 if(bpp==1) mask=0x1;
 else
 if(bpp==2) mask=0x3;
 else
 if(bpp==4) mask=0xF;
 else       mask=0xFF;


 /* scan the image, calculate histogram */

 for(y=0;y<ypix;y++)
 {
  err=imfind1r(im,y,&data);
  if(err) break;
  p=(char*)data;

  if(bpp<=8)
  {
   c=*data++;
   shift=0;

   for(x=0;x<xpix;x++)
   {
    index=(c>>shift)&mask;
    shift+=bpp;
    if(shift>=32)
    {
     shift=0;
     c=*data++;
    }

    rr=rtab[index];
    gg=gtab[index];
    bb=btab[index];
    rf[rr]++;
    gf[gg]++;
    bf[bb]++;
    Histogram[(((rr<<bits)|gg)<<bits)|bb]++;
   }
  }
  else
  {
   if(bpp==24)
   {
    for(x=0;x<xpix;x++)
    {
     rr=(*p++)>>(8-bits);
     gg=(*p++)>>(8-bits);
     bb=(*p++)>>(8-bits);
     rf[rr]++;
     gf[gg]++;
     bf[bb]++;
     Histogram[(((rr<<bits)|gg)<<bits)|bb]++;
    }
   }
   else
   if(bpp==32)
   {
    mask=(1<<bits)-1;

    for(x=0;x<xpix;x++)
    {
     c=*data++;

     rr=(c>>(8-bits)) & mask;
     gg=(c>>(8+8-bits)) & mask;
     bb=(c>>(16+8-bits)) & mask;

     rf[rr]++;
     gf[gg]++;
     bf[bb]++;
     Histogram[(((rr<<bits)|gg)<<bits)|bb]++;
    }
   }
   else
   if(bpp==16)
   {
    c=*data++;
    shift=0;

    for(x=0;x<xpix;x++)
    {
     if(shift) rr=(c>>(shift-3))&0xF8;
     else      rr=(c<<3)&0xF8;

     gg=(c>>(shift+2))&0xF8;
     bb=(c>>(shift+7))&0xF8;

     rr=rr>>(8-bits);
     gg=gg>>(8-bits);
     bb=bb>>(8-bits);

     rf[rr]++;
     gf[gg]++;
     bf[bb]++;
     Histogram[(((rr<<bits)|gg)<<bits)|bb]++;

     shift+=bpp;
     if(shift>=32)
     {
      c=*data++;
      shift=0;
     }
    }
   }
  }
 }           
 return(err);
}






/*
 * Perform variance-based color quantization on a 24-bit image.
 *
 * Input consists of:
 *      red, green, blue        Arrays of red, green and blue pixel
 *                              intensities stored as unsigned characters.
 *                              The color of the ith pixel is given
 *                              by red[i] green[i] and blue[i].  0 indicates
 *                              zero intensity, 255 full intensity.
 *      pixels                  The length of the red, green and blue arrays
 *                              in bytes, stored as an unsigned long int.
 *      colormap                Points to the colormap.  The colormap
 *                              consists of red, green and blue arrays.
 *                              The red/green/blue values of the ith
 *                              colormap entry are given respectively by
 *                              colormap[0][i], colormap[1][i] and
 *                              colormap[2][i].  Each entry is an unsigned char.
 *      colors                  The number of colormap entries, stored
 *                              as an integer.  
 *      bits                    The number of significant bits in each entry
 *                              of the red, greena and blue arrays. An integer.
 *      rgbmap                  An array of unsigned chars of size (2^bits)^3.
 *                              This array is used to map from pixels to
 *                              colormap entries.  The 'prequantized' red,
 *                              green and blue components of a pixel
 *                              are used as an index into rgbmap to retrieve
 *                              the index which should be used into the colormap
 *                              to represent the pixel.  In short:
 *                              index = rgbmap[(((r << bits) | g) << bits) | b];
 *      flags                   A set of bit-flags:
 *              CQ_FAST         If set, the rgbmap will be constructed
 *                              quickly.  If not set, the rgbmap will
 *                              be built more slowly, but more
 *                              accurately.  In most cases, fast
 *                              should be set, as the error introduced
 *                              by the approximation is usually small.
 *              CQ_QUANTIZE:    If set, the data in red, green, and
 *                              blue is not pre-quantized, and will be
 *                              quantized "on the fly".  This slows
 *                              the routine slightly.  If not set, the
 *                              data has been prequantized to 'bits'
 *                              significant bits.
 *              
 *      accum_hist              If non-zero the histogram will accumulate and 
 *                              reflect pixels from multiple images.
 *                              when 1, the histogram will be initialized and
 *                              summed, but not thrown away OR processed. when 
 *                              2 the image RGB will be added to it.  When 3 
 *                              Boxes are cut and a colormap and rgbmap
 *                              are be returned, Histogram is freed too.
 *                              When zero, all code is executed as per normal.
 *
 * colorquant returns the number of colors to which the image was
 * quantized.
 */


/* bits is how many bits of input is significant */

os_error * colorquant(imagestr * sim,imagestr * dim,int bits)
{
 os_error * err;
 int        i;                      /* Counter */
 int        OutColors;              /* # of entries computed */
 int        Colormax;               /* quantized full-intensity */ 
 double     Cfactor;                /* Conversion factor */
 int        colors;
 int        bpp;
 unsigned char r;
 unsigned char g;
 unsigned char b;


 ColormaxI=1<<bits;      /* 2 ^ Bits */
 Colormax=ColormaxI-1;
 Bits=bits;
 NPixels=sim->xpix*sim->ypix;
 bpp=dim->bpp;
 Cfactor=(double)FULLINTENSITY/Colormax;
 colors=(1<<bpp);
 dim->ipal.ncolours=colors;

 err=flex_alloc((flex_ptr)&Histogram,
                         ColormaxI*ColormaxI*ColormaxI*sizeof(int));
 if(!err)
 {
  memset(Histogram,0,ColormaxI*ColormaxI*ColormaxI*sizeof(int));

  err=flex_alloc((flex_ptr)&Boxes,colors*sizeof(Box));
  if(!err)
  {
   memset(Boxes->freq[0],0,ColormaxI*sizeof(unsigned int));
   memset(Boxes->freq[1],0,ColormaxI*sizeof(unsigned int));
   memset(Boxes->freq[2],0,ColormaxI*sizeof(unsigned int));

   SumPixels=NPixels;

   err=calchist(sim,&Boxes[0],bits);
   if(!err)
   {
    OutColors=splitboxes(Boxes,colors);

        /*
         * We now know the set of representative colors.  We now
         * must fill in the colormap and convert the representatives
         * from their 'prequantized' range to 0-FULLINTENSITY.
         */

    for(i=0;i<OutColors;i++) 
    {
     r=(unsigned char)(Boxes[i].mean[REDI] * Cfactor + 0.5);
     g=(unsigned char)(Boxes[i].mean[GREENI] * Cfactor + 0.5);
     b=(unsigned char)(Boxes[i].mean[BLUEI] * Cfactor + 0.5);

     dim->ipal.word[i]=(b<<24)|(g<<16)|(r<<8);
    }
   }
   flex_free((flex_ptr)&Boxes);
  }
  flex_free((flex_ptr)&Histogram);
 }
 return(err);
}

